#!/bin/sh

# Keluar segera jika sebuah perintah gagal.
set -e
# Anggap variabel yang tidak di-set sebagai error.
set -u

# --- Variabel Konfigurasi ---
BIN_DIR="/usr/bin"
CONFIG_DIR="/etc/noobzvpns"
SYSTEMD_DIR="/etc/systemd/system"
SERVICE_NAME="noobzvpns.service"
BINARY_NAME="noobzvpns"

# --- Variabel Dinamis ---
# Dapatkan direktori tempat skrip ini berada
SCRIPT_DIR="$(dirname "$0")"
# Dapatkan arsitektur mesin
MACHINE_ARCH="$(uname -m)"
SYSTEMCTL_PATH="$(which systemctl)"
BINARY_ARCH_NAME=""

# --- Fungsi ---

# Fungsi untuk menampilkan pesan error dan keluar
die() {
    echo "Error: $1" >&2
    exit 1
}

# 1. Memeriksa semua prasyarat instalasi
check_prerequisites() {
    echo "1. Memeriksa prasyarat..."

    if [ "$(id -u)" -ne 0 ]; then
        die "Skrip ini harus dijalankan sebagai root."
    fi

    case "$MACHINE_ARCH" in
        "x86_64")
            BINARY_ARCH_NAME="noobzvpns.x86_64"
            ;;
        *)
            die "Instalasi gagal, arsitektur CPU '$MACHINE_ARCH' tidak didukung."
            ;;
    esac
    echo "   - Arsitektur CPU: $MACHINE_ARCH, Biner: $BINARY_ARCH_NAME"

    if [ ! -d "$SYSTEMD_DIR" ] || [ ! -f "$SYSTEMCTL_PATH" ]; then
        die "Systemd tidak ditemukan. Pastikan distro Anda menggunakan systemd sebagai init default."
    fi
    echo "   - Prasyarat terpenuhi."
}

# 2. Membersihkan instalasi lama (jika ada)
cleanup_old_install() {
    echo "2. Mempersiapkan upgrade/instalasi (membersihkan versi lama)..."
    if [ -f "$SYSTEMD_DIR/$SERVICE_NAME" ]; then
        # Menggunakan '|| true' agar tidak gagal jika service tidak berjalan
        "$SYSTEMCTL_PATH" stop "$SERVICE_NAME" || true
        "$SYSTEMCTL_PATH" disable "$SERVICE_NAME" || true
    fi
    # Menggunakan 'rm -f' untuk menghindari error jika file tidak ada
    rm -f "$SYSTEMD_DIR/$SERVICE_NAME"
    rm -f "$BIN_DIR/$BINARY_NAME"
    rm -f "$CONFIG_DIR/cert.pem"
    rm -f "$CONFIG_DIR/key.pem"
}

# 3. Menyalin file-file baru ke direktori tujuan
install_files() {
    echo "3. Menyalin file biner dan konfigurasi..."
    # Menggunakan 'mkdir -p' untuk membuat direktori jika belum ada
    mkdir -p "$CONFIG_DIR"
    cp "$SCRIPT_DIR/$BINARY_ARCH_NAME" "$BIN_DIR/$BINARY_NAME"
    cp "$SCRIPT_DIR/cert.pem" "$CONFIG_DIR/cert.pem"
    cp "$SCRIPT_DIR/key.pem" "$CONFIG_DIR/key.pem"
    cp "$SCRIPT_DIR/$SERVICE_NAME" "$SYSTEMD_DIR/$SERVICE_NAME"

    # Hanya salin config.json jika belum ada
    if [ ! -f "$CONFIG_DIR/config.json" ]; then
        echo "   - File 'config.json' tidak ditemukan, menyalin template default."
        cp "$SCRIPT_DIR/config.json" "$CONFIG_DIR/config.json"
    fi
}

# 4. Mengatur izin akses (permission) file
set_permissions() {
    echo "4. Mengatur izin akses file..."
    chmod 700 "$BIN_DIR/$BINARY_NAME"
    chmod 600 "$CONFIG_DIR/cert.pem"
    chmod 600 "$CONFIG_DIR/key.pem"
    chmod 600 "$CONFIG_DIR/config.json"
    chmod 600 "$SYSTEMD_DIR/$SERVICE_NAME"
}

# 5. Menyelesaikan instalasi dengan systemd
finish_install() {
    echo "5. Menyelesaikan instalasi..."
    "$SYSTEMCTL_PATH" daemon-reload
    "$SYSTEMCTL_PATH" enable "$SERVICE_NAME"
}

# Menampilkan instruksi akhir kepada pengguna
print_instructions() {
    echo ""
    echo "Upgrade/Instalasi selesai."
    echo ""
# Menggunakan Here Document untuk output multi-baris yang bersih
cat <<EOF
[ CATATAN ]
- Lokasi Konfigurasi & SSL (KEY/CERT) : $CONFIG_DIR
- Perintah Service :
   + Mulai       -> systemctl start $SERVICE_NAME
   + Ulang       -> systemctl restart $SERVICE_NAME
   + Hentikan    -> systemctl stop $SERVICE_NAME
   + Status/Log  -> systemctl status $SERVICE_NAME -l
   + Aktifkan    -> systemctl enable $SERVICE_NAME (sudah aktif secara default)
   + Nonaktifkan -> systemctl disable $SERVICE_NAME
   + Log penuh   -> journalctl -u $SERVICE_NAME
EOF
}

# --- Alur Eksekusi Utama ---
main() {
    echo
    check_prerequisites
    cleanup_old_install
    install_files
    set_permissions
    finish_install
    print_instructions
    echo
}

# Jalankan fungsi main
main
